/// <reference types="node" />
import { Application } from "express";
import { EventEmitter } from "events";
export declare interface AppService {
    /**
     * Emitted when an event is pushed to the appservice.
     * The format of the event object is documented at
     * https://matrix.org/docs/spec/application_service/r0.1.2#put-matrix-app-v1-transactions-txnid
     * @event
     * @example
     * appService.on("event", function(ev) {
     *   console.log("ID: %s", ev.event_id);
     * });
     */
    on(event: "event", cb: (event: Record<string, unknown>) => void): this;
    /**
     * Emitted when an ephemeral event is pushed to the appservice.
     * The format of the event object is documented at
     * https://github.com/matrix-org/matrix-doc/pull/2409
     * @event
     * @example
     * appService.on("ephemeral", function(ev) {
     *   console.log("ID: %s", ev.type);
     * });
     */
    on(event: "ephemeral", cb: (event: Record<string, unknown>) => void): this;
    /**
     * Emitted when the HTTP listener logs some information.
     * `access_tokens` are stripped from requests
     * @event
     * @example
     * appService.on("http-log", function(line) {
     *   console.log(line);
     * });
     */
    on(event: "http-log", cb: (line: string) => void): this;
    /**
     * Emitted when an event of a particular type is pushed
     * to the appservice. This will be emitted *in addition*
     * to "event", so ensure your bridge deduplicates events.
     * @event
     * @param event Should start with "type:"
     * @example
     * appService.on("type:m.room.message", function(event) {
     *   console.log("ID: %s", ev.content.body);
     * });
     */
    on(event: string, cb: (event: Record<string, unknown>) => void): this;
}
export declare class AppService extends EventEmitter {
    private config;
    /**
     * @deprecated Use `AppService.expressApp`
     */
    readonly app: Application;
    private server?;
    private lastProcessedTxnId;
    /**
     * Construct a new application service.
     * @constructor
     * @param {Object} config Configuration for this service.
     * @param {String} config.homeserverToken The incoming HS token to expect. Must
     * be set prior to calling listen(port).
     * @param {Number} config.httpMaxSizeBytes The max number of bytes allowed on an
     * incoming HTTP request. Default: 5000000.
     * @throws If a homeserver token is not supplied.
     */
    constructor(config: {
        homeserverToken: string;
        httpMaxSizeBytes?: number;
    });
    /***
     * Begin listening on the specified port.
     * @param {Number} port The port to listen on.
     * @param {String} hostname Optional hostname to listen on
     * @param {Number} backlog Maximum length of the queue of pending connections
     * @param {Function} callback The callback for the "listening" event. Optional.
     * @returns {Promise} When the server is listening, if a callback is not provided.
     */
    listen(port: number, hostname: string, backlog: number, callback?: () => void): Promise<void> | undefined;
    /**
     * Closes the HTTP server.
     * @returns {Promise} When the operation has completed
     * @throws If the server has not been started.
     */
    close(): Promise<void>;
    /**
     * Override this method to handle alias queries.
     * @param {string} alias The queried room alias
     * @param {Function} callback The callback to invoke when complete.
     * @return {Promise} A promise to resolve when complete (if callback isn't supplied)
     */
    onAliasQuery(alias: string, callback: () => void): PromiseLike<void> | null;
    /**
     * Override this method to handle user queries.
     * @param {string} userId The queried user ID.
     * @param {Function} callback The callback to invoke when complete.
     * @return {Promise} A promise to resolve when complete (if callback isn't supplied)
     */
    onUserQuery(userId: string, callback: () => void): PromiseLike<void> | null;
    /**
     * Set the token that should be used to verify incoming events.
     * @param {string} hsToken The home server token
     */
    setHomeserverToken(hsToken: string): void;
    /**
     * The Express App instance for the appservice, which
     * can be extended with paths.
     */
    get expressApp(): Application;
    private onMorganLog;
    private isInvalidToken;
    private onGetUsers;
    private onGetRoomAlias;
    private onTransaction;
    private onHealthCheck;
}
